////////////////////////////////////////////////////////////////////////////////
//
// Copyright (c) since 1999. VASCO DATA SECURITY
// All rights reserved. http://www.vasco.com
//
////////////////////////////////////////////////////////////////////////////////

#import <Foundation/Foundation.h>

@class InputError;
@protocol OrchestrationProtocol;
@protocol UserAuthenticationInputDelegate;

NS_ASSUME_NONNULL_BEGIN

/**
 * <p>The {@link UserAuthenticationDelegate} interface defines the specific elements required for the integrator to interact during a User Authentication flow.</p>
 * <p>If the integrator use a custom user authentication, this interface will be call during any flow where a User Authentication is needed.</p>
 *
 * <p>A {@link UserAuthenticationDelegate} object should be provided to the {@link Orchestrator} object
 * in order to receive feedback during the user authentication process.</p>
*/
NS_REFINED_FOR_SWIFT
@protocol UserAuthenticationDelegate <NSObject>

/**
 * User authentication type that can be customized by the integrator
*/

typedef NS_ENUM(NSInteger, UserAuthenticationType) {
    /** The User is expected to authenticate with a Password (or a PIN) */
    UserAuthenticationTypePassword,
    
    /** The User is expected to authenticate with a Biometric method  */
    UserAuthenticationTypeBiometric
} NS_REFINED_FOR_SWIFT;

/**
 * Called when a specific user authentication have been overridden and if an user authentication is required by the Orchestration library.
 * A authentication should be presented to the user. And the user input should be returned throw the inputCallback.
 *
 * @param type {@link UserAuthentication} the type of authentication asked by the Orchestration library
 * @param inputCallback a {@link UserAuthenticationInputDelegate} delegate to give the user input to the Orchestration library
 * @param isEnrollment is set to true in case of enrollment
 */
- (void)orchestrator:(id <OrchestrationProtocol>)orchestrator onUserAuthenticationRequired:(UserAuthenticationType)type inputDelegate:(id <UserAuthenticationInputDelegate>)inputDelegate isEnrollment:(BOOL)isEnrollment;

/**
 * Called upon user input error when it is not conform for the Orchestration library.
 *
 * @param error the type of input error
*/
- (void)orchestrator:(id <OrchestrationProtocol>)orchestrator onUserAuthenticationInputError:(InputError *)error;

@end

NS_ASSUME_NONNULL_END
